# X-LoRA

Mixture of LoRA Experts ([X-LoRA](https://huggingface.co/papers/2402.07148)) is a PEFT method enabling sparse or dense mixture of LoRA experts based on a high granularity (token, layer, sequence) scalings matrix. This leverages frozen LoRA adapters and a frozen base model to drastically reduces the number of parameters that need to be fine-tuned.

A unique aspect of X-LoRA is its versatility: it can be applied to any `transformers` base model with LoRA adapters. This means that, despite the mixture of experts strategy, no changes to the model code must be made.

The below graphic demonstrates how the scalings change for different prompts for each token. This highlights the activation of different adapters as the generation progresses and the sequence creates new context.

![Token-by-token scalings](https://github.com/EricLBuehler/xlora/raw/master/res/token_by_token_scalings.gif)

The abstract from the paper is:

*We report a mixture of expert strategy to create fine-tuned large language models using a deep layer-wise token-level approach based on low-rank adaptation (LoRA). Starting with a set of pre-trained LoRA adapters, our gating strategy uses the hidden states to dynamically mix adapted layers, allowing the resulting X-LoRA model to draw upon different capabilities and create never-before-used deep layer-wise combinations to solve tasks. The design is inspired by the biological principles of universality and diversity, where neural network building blocks are reused in different hierarchical manifestations. Hence, the X-LoRA model can be easily implemented for any existing large language model (LLM) without a need for modifications of the underlying structure. We develop a tailored X-LoRA model that offers scientific capabilities including forward/inverse analysis tasks and enhanced reasoning capability, focused on biomaterial analysis, protein mechanics and design. The impact of this work include access to readily expandable and adaptable models with strong domain knowledge and the capability to integrate across areas of knowledge. Featuring experts in biology, mathematics, reasoning, bio-inspired materials, mechanics and materials, chemistry, protein biophysics, mechanics and quantum-mechanics based molecular properties, we conduct a series of physics-focused case studies. We examine knowledge recall, protein mechanics forward/inverse tasks, protein design, adversarial agentic modeling including ontological knowledge graph construction, as well as molecular design. The model is capable not only of making quantitative predictions of nanomechanical properties of proteins or quantum mechanical molecular properties, but also reasons over the results and correctly predicts likely mechanisms that explain distinct molecular behaviors.*.

Please cite X-LoRA as:
```bibtex
@article{10.1063/5.0203126,
    author = {Buehler, Eric L. and Buehler, Markus J.},
    title = "{X-LoRA: Mixture of low-rank adapter experts, a flexible framework for large language models with applications in protein mechanics and molecular design}",
    journal = {APL Machine Learning},
    volume = {2},
    number = {2},
    pages = {026119},
    year = {2024},
    month = {05},
    abstract = "{We report a mixture of expert strategy to create fine-tuned large language models using a deep layer-wise token-level approach based on low-rank adaptation (LoRA). Starting with a set of pre-trained LoRA adapters, our gating strategy uses the hidden states to dynamically mix adapted layers, allowing the resulting X-LoRA model to draw upon different capabilities and create never-before-used deep layer-wise combinations to solve tasks. The design is inspired by the biological principles of universality and diversity, where neural network building blocks are reused in different hierarchical manifestations. Hence, the X-LoRA model can be easily implemented for any existing large language model without a need for modifications of the underlying structure. We develop a tailored X-LoRA model that offers scientific capabilities, including forward/inverse analysis tasks and enhanced reasoning capability, focused on biomaterial analysis, protein mechanics, and design. The impact of this work includes access to readily expandable and adaptable models with strong domain knowledge and the capability to integrate across areas of knowledge. Featuring experts in biology, mathematics, reasoning, bio-inspired materials, mechanics and materials, chemistry, protein biophysics, mechanics, and quantum-mechanics based molecular properties, we conduct a series of physics-focused case studies. We examine knowledge recall, protein mechanics forward/inverse tasks, protein design, adversarial agentic modeling including ontological knowledge graph construction, and molecular design. The model is capable not only of making quantitative predictions of nanomechanical properties of proteins or quantum mechanical molecular properties but also reasoning over the results and correctly predicting likely mechanisms that explain distinct molecular behaviors.}",
    issn = {2770-9019},
    doi = {10.1063/5.0203126},
    url = {https://doi.org/10.1063/5.0203126},
    eprint = {https://pubs.aip.org/aip/aml/article-pdf/doi/10.1063/5.0203126/19964043/026119\_1\_5.0203126.pdf},
}
```

## XLoraConfig[[peft.XLoraConfig]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class peft.XLoraConfig</name><anchor>peft.XLoraConfig</anchor><source>https://github.com/huggingface/peft/blob/main/src/peft/tuners/xlora/config.py#L25</source><parameters>[{"name": "task_type", "val": ": Optional[Union[str, TaskType]] = None"}, {"name": "peft_type", "val": ": Optional[Union[str, PeftType]] = None"}, {"name": "auto_mapping", "val": ": Optional[dict] = None"}, {"name": "peft_version", "val": ": Optional[str] = None"}, {"name": "base_model_name_or_path", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "inference_mode", "val": ": bool = False"}, {"name": "hidden_size", "val": ": int = None"}, {"name": "adapters", "val": ": dict[str, str] = None"}, {"name": "enable_softmax", "val": ": bool = True"}, {"name": "enable_softmax_topk", "val": ": bool = False"}, {"name": "layerwise_scalings", "val": ": bool = False"}, {"name": "xlora_depth", "val": ": int = 1"}, {"name": "xlora_size", "val": ": int = 2048"}, {"name": "xlora_dropout_p", "val": ": float = 0.2"}, {"name": "use_trainable_adapters", "val": ": bool = False"}, {"name": "softmax_temperature", "val": ": float = 1.0"}, {"name": "top_k_lora", "val": ": Optional[int] = None"}, {"name": "scaling_pass_value", "val": ": float = 0.0"}, {"name": "global_scaling_weight", "val": ": float = 1.0"}]</parameters><paramsdesc>- **hidden_size** (`int`) --
  Hidden size of the base model.
- **adapters** (`dict`) --
  Mapping of adapter names to the LoRA adapter id, as per PeftModel.load_adapter. *They will be automatically
  loaded*, to use as LoRA experts. When using from_pretrained, pass the new adapters dict as a keyword
  argument.
- **enable_softmax** (`bool`, *optional*, defaults to `True`) --
  Enable softmax application for the X-LoRA classifier.
- **enable_softmax_topk** (`bool`, *optional*, defaults to `False`) --
  Enable softmax application for the top-k LoRA adapters. Mutually exclusive to `enable_softmax` and must
  only be set if `top_k_lora` is.
- **softmax_temperature** (`float`, *optional*, defaults to 1.0) --
  Softmax temperature, lower yields sharper predictions
- **layerwise_scalings** (`bool`, *optional*, defaults to `False`) --
  If True, generate scalings for each LoRA adapter (each layer). If this is False, then scalings will be
  broadcasted, the same, to each layer.
- **top_k_lora** (`int`, *optional*, defaults to None) --
  Sparsely select the top_k LoRA experts instead of the default dense method.
- **xlora_depth** (`int`, *optional*, defaults to 1) --
  Depth of the X-LoRA classifier.
- **xlora_size** (`int`, *optional*, defaults to 2048) --
  Hidden size of the X-LoRA classifier, irrelevant if `xlora_depth=1`.
- **xlora_dropout_p** (`float`, *optional*, defaults to 0.2) --
  Dropout probability of the X-LoRA classifier, irrelevant if `xlora_depth=1`.
- **use_trainable_adapters** (`bool`, *optional*, defaults to False) --
  Make the adapters trainable.
- **scaling_pass_value** (`float`, *optional*, defaults to 0) --
  Scaling pass value.
- **global_scaling_weight** (`float`, *optional*, defaults to 1) --
  Weight to multiply output of each LoRA adapter by.</paramsdesc><paramgroups>0</paramgroups></docstring>

This is the configuration class to store the configuration of a `XLoraModel`. When the config is reloaded, the
paths of the `adapters` field is disregarded in favor of the saved adapters. As such, only the keys matter during
loading.




</div>

## XLoraModel[[peft.XLoraModel]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class peft.XLoraModel</name><anchor>peft.XLoraModel</anchor><source>https://github.com/huggingface/peft/blob/main/src/peft/tuners/xlora/model.py#L156</source><parameters>[{"name": "model", "val": ": nn.Module"}, {"name": "config", "val": ": Union[dict[str, XLoraConfig], XLoraConfig]"}, {"name": "adapter_name", "val": ": str"}, {"name": "torch_device", "val": ": Optional[str] = None"}, {"name": "ephemeral_gpu_offload", "val": ": bool = False"}, {"name": "autocast_adapter_dtype", "val": ": bool = True"}, {"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **model** (`torch.nn.Module`) -- The model to be adapted.
- **config** ([XLoraConfig](/docs/peft/main/en/package_reference/xlora#peft.XLoraConfig)) -- The configuration of the Lora model.
- **adapter_name** (`str`) -- The name of the adapter, does not affect the LoRA adapter names.</paramsdesc><paramgroups>0</paramgroups><rettype>`torch.nn.Module`</rettype><retdesc>The X-LoRA model.</retdesc></docstring>

Creates an X-LoRA (Mixture of LoRA experts), model from a pretrained transformers model. Currently, this X-LoRA
implementation only works with models with a transformer architecture.

The method is described in detail in https://huggingface.co/papers/2402.07148.







<ExampleCodeBlock anchor="peft.XLoraModel.example">

Example:
```py
>>> from transformers import AutoModelForCausalLM, AutoConfig, BitsAndBytesConfig
>>> from peft import LoraConfig, PeftModel, get_peft_model, prepare_model_for_kbit_training

>>> model_config = AutoConfig.from_pretrained("mistralai/Mistral-7B-Instruct-v0.1")
>>> config = XLoraConfig(
...     task_type="CAUSAL_LM",
...     hidden_size=model_config.hidden_size,
...     xlora_depth=4,
...     adapters={
...         "adapter_1": "./path/to/the/checkpoint/",
...         "adapter_2": "./path/to/the/checkpoint/",
...         "adapter_n": "./path/to/the/checkpoint/",
...     },
... )
>>> int8_config = BitsAndBytesConfig(load_in_8bit=True)
>>> model = AutoModelForCausalLM.from_pretrained(
...     "mistralai/Mistral-7B-Instruct-v0.1",
...     trust_remote_code=True,
...     attn_implementation="flash_attention_2",
...     device_map="cuda:0",
...     torch_dtype=torch.bfloat16,
...     quantization_config=int8_config,
... )
>>> model = prepare_model_for_kbit_training(4)
>>> xlora_model = get_peft_model(model, config)
```

</ExampleCodeBlock>



<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>clear_scalings_log</name><anchor>peft.XLoraModel.clear_scalings_log</anchor><source>https://github.com/huggingface/peft/blob/main/src/peft/tuners/xlora/model.py#L510</source><parameters>[]</parameters></docstring>

Clear the scalings log.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>disable_scalings_logging</name><anchor>peft.XLoraModel.disable_scalings_logging</anchor><source>https://github.com/huggingface/peft/blob/main/src/peft/tuners/xlora/model.py#L503</source><parameters>[]</parameters></docstring>

Disable scalings logging, without clearing the log.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>enable_scalings_logging</name><anchor>peft.XLoraModel.enable_scalings_logging</anchor><source>https://github.com/huggingface/peft/blob/main/src/peft/tuners/xlora/model.py#L496</source><parameters>[]</parameters></docstring>

Enable scalings logging.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_bucketed_scalings_log</name><anchor>peft.XLoraModel.get_bucketed_scalings_log</anchor><source>https://github.com/huggingface/peft/blob/main/src/peft/tuners/xlora/model.py#L517</source><parameters>[]</parameters></docstring>

Returns bucketed scalings, bucketed by seq_len. Each value consists of the positions (the first) and the
associated tensors. The positions are paired with the associated tensors and give the position in the scaling
log.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_global_scaling_weight</name><anchor>peft.XLoraModel.get_global_scaling_weight</anchor><source>https://github.com/huggingface/peft/blob/main/src/peft/tuners/xlora/model.py#L473</source><parameters>[]</parameters></docstring>

Get the global LoRA weight.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_latest_scalings</name><anchor>peft.XLoraModel.get_latest_scalings</anchor><source>https://github.com/huggingface/peft/blob/main/src/peft/tuners/xlora/model.py#L480</source><parameters>[]</parameters></docstring>

Returns the latest scalings prediction, or None if no scalings have been predicted. The tensor is of shape
(batch_size, seq_len, n_layers, n_classes).


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_scalings_log</name><anchor>peft.XLoraModel.get_scalings_log</anchor><source>https://github.com/huggingface/peft/blob/main/src/peft/tuners/xlora/model.py#L487</source><parameters>[]</parameters></docstring>

Returns a shallow (only copying the list itself not the tensors) copy of the list containing the scalings log.
Editing the list does not change the underlying log. The tensors are of shape (batch_size, seq_len, n_layers,
n_classes). The seq_len dim may vary with input dimension.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>set_global_scaling_weight</name><anchor>peft.XLoraModel.set_global_scaling_weight</anchor><source>https://github.com/huggingface/peft/blob/main/src/peft/tuners/xlora/model.py#L457</source><parameters>[{"name": "weight", "val": ": float"}]</parameters></docstring>

Set the global LoRA weight, a scalar to multiply the output of each LoRA adapter by. This is by default 1. This
is reflected in the config.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>set_scaling_pass_value</name><anchor>peft.XLoraModel.set_scaling_pass_value</anchor><source>https://github.com/huggingface/peft/blob/main/src/peft/tuners/xlora/model.py#L465</source><parameters>[{"name": "value", "val": ": float | None"}]</parameters></docstring>

Set the scaling pass value, the value to set the scalings to during the scaling pass. If the value is None, the
scaling pass value will be 1/n where n is the number of adapters.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>set_topk_lora</name><anchor>peft.XLoraModel.set_topk_lora</anchor><source>https://github.com/huggingface/peft/blob/main/src/peft/tuners/xlora/model.py#L449</source><parameters>[{"name": "value", "val": ": Optional[int]"}]</parameters></docstring>

Sparsely select the specified top_k LoRA experts instead of the default dense method. Set to None to use dense.
This is reflected in the config.


</div></div>

<EditOnGithub source="https://github.com/huggingface/peft/blob/main/docs/source/package_reference/xlora.md" />