# States

Blocks rely on the [PipelineState](/docs/diffusers/main/en/api/modular_diffusers/pipeline_states#diffusers.modular_pipelines.PipelineState) and [BlockState](/docs/diffusers/main/en/api/modular_diffusers/pipeline_states#diffusers.modular_pipelines.BlockState) data structures for communicating and sharing data.

| State | Description |
|-------|-------------|
| [PipelineState](/docs/diffusers/main/en/api/modular_diffusers/pipeline_states#diffusers.modular_pipelines.PipelineState) | Maintains the overall data required for a pipeline's execution and allows blocks to read and update its data. |
| [BlockState](/docs/diffusers/main/en/api/modular_diffusers/pipeline_states#diffusers.modular_pipelines.BlockState) | Allows each block to perform its computation with the necessary data from `inputs`|

This guide explains how states work and how they connect blocks.

## PipelineState

The [PipelineState](/docs/diffusers/main/en/api/modular_diffusers/pipeline_states#diffusers.modular_pipelines.PipelineState) is a global state container for all blocks. It maintains the complete runtime state of the pipeline and provides a structured way for blocks to read from and write to shared data.

There are two dict's in [PipelineState](/docs/diffusers/main/en/api/modular_diffusers/pipeline_states#diffusers.modular_pipelines.PipelineState) for structuring data.

- The `values` dict is a **mutable** state containing a copy of user provided input values and intermediate output values generated by blocks. If a block modifies an `input`, it will be reflected in the `values` dict after calling `set_block_state`.

```py
PipelineState(
  values={
    'prompt': 'a cat'
    'guidance_scale': 7.0
    'num_inference_steps': 25
    'prompt_embeds': Tensor(dtype=torch.float32, shape=torch.Size([1, 1, 1, 1]))
    'negative_prompt_embeds': None
  },
)
```

## BlockState

The [BlockState](/docs/diffusers/main/en/api/modular_diffusers/pipeline_states#diffusers.modular_pipelines.BlockState) is a local view of the relevant variables an individual block needs from [PipelineState](/docs/diffusers/main/en/api/modular_diffusers/pipeline_states#diffusers.modular_pipelines.PipelineState) for performing it's computations.

Access these variables directly as attributes like `block_state.image`.

```py
BlockState(
    image: 
)
```

When a block's `__call__` method is executed, it retrieves the `BlockState` with `self.get_block_state(state)`, performs it's operations, and updates [PipelineState](/docs/diffusers/main/en/api/modular_diffusers/pipeline_states#diffusers.modular_pipelines.PipelineState) with `self.set_block_state(state, block_state)`.

```py
def __call__(self, components, state):
    # retrieve BlockState
    block_state = self.get_block_state(state)

    # computation logic on inputs

    # update PipelineState
    self.set_block_state(state, block_state)
    return components, state
```

## State interaction

[PipelineState](/docs/diffusers/main/en/api/modular_diffusers/pipeline_states#diffusers.modular_pipelines.PipelineState) and [BlockState](/docs/diffusers/main/en/api/modular_diffusers/pipeline_states#diffusers.modular_pipelines.BlockState) interaction is defined by a block's `inputs`, and `intermediate_outputs`.

- `inputs`, a block can modify an input - like `block_state.image` - and this change can be propagated globally to [PipelineState](/docs/diffusers/main/en/api/modular_diffusers/pipeline_states#diffusers.modular_pipelines.PipelineState) by calling `set_block_state`.
- `intermediate_outputs`, is a new variable that a block creates. It is added to the [PipelineState](/docs/diffusers/main/en/api/modular_diffusers/pipeline_states#diffusers.modular_pipelines.PipelineState)'s `values` dict and is available as for subsequent blocks or accessed by users as a final output from the pipeline.

